/*
 * @brief Secondary loader main application code
 *
 * @note
 * Copyright(C) NXP Semiconductors, 2014
 * All rights reserved.
 *
 * @par
 * Software that is described herein is for illustrative purposes only
 * which provides customers with programming information regarding the
 * LPC products.  This software is supplied "AS IS" without any warranties of
 * any kind, and NXP Semiconductors and its licensor disclaim any and
 * all warranties, express or implied, including all implied warranties of
 * merchantability, fitness for a particular purpose and non-infringement of
 * intellectual property rights.  NXP Semiconductors assumes no responsibility
 * or liability for the use of the software, conveys no license or rights under any
 * patent, copyright, mask work right, or any other intellectual property rights in
 * or to any products. NXP Semiconductors reserves the right to make changes
 * in the software without notification. NXP Semiconductors also makes no
 * representation or warranty that such application will be suitable for the
 * specified use without further testing or modification.
 *
 * @par
 * Permission to use, copy, modify, and distribute this software and its
 * documentation is hereby granted, under NXP Semiconductors' and its
 * licensor's relevant copyrights in the software, without fee, provided that it
 * is used in conjunction with NXP Semiconductors microcontrollers.  This
 * copyright, permission, and disclaimer notice must appear in all copies of
 * this code.
 */

#include "chip.h"
#include "sl_common.h"
#include "LPC865.h"
#include "pin_mux.h"
#include "sl_protocol.h"
#include "board.h"
#include "fsl_iap.h"


/*****************************************************************************
 * Private types/enumerations/variables
 ****************************************************************************/
extern spi_transfer_t spiSlaveXfer;
extern int buffXferPending;

typedef void (*loopHostIfFunc)(int pNum);
loopHostIfFunc loopI2C_pFunc;

/*****************************************************************************
 * Public types/enumerations/variables
 ****************************************************************************/
 SL_HOSTIF_PACKET_T sendBuff, recvBuff;

IMG_HEADER_T *pImageHeader1 = (IMG_HEADER_T *) (SL_BOOTAPP_ADDR1 + SL_BOOTAPP_IMGHDR_OFS);
IMG_HEADER_T *pImageHeader2 = (IMG_HEADER_T *) (SL_BOOTAPP_ADDR2 + SL_BOOTAPP_IMGHDR_OFS);

SL_PINSETUP_T *pinCfgData = (SL_PINSETUP_T *) SL_ADDRESS_APPPINDATA;
static uint32_t *pushAppFlag = (uint32_t *) SL_ADDRESS_APPCALLEDFL;

uint32_t *FW_VERSION1 = (uint32_t *) (SL_BOOTAPP_ADDR1 + SL_FW_VERSION_OFFSET);
uint32_t *FW_VERSION2 = (uint32_t *) (SL_BOOTAPP_ADDR2 + SL_FW_VERSION_OFFSET);

uint8_t App_CRC_Flag1 = 0;
uint8_t App_CRC_Flag2 = 0;

/*****************************************************************************
 * Private functions
 
 ****************************************************************************/

void GPIOSetDir( uint32_t portNum, uint32_t bitPosi, uint32_t dir )
{
    if( dir ) {
        GPIO->DIRSET[portNum] = 1<<bitPosi;
    }
    else {
        GPIO->DIRCLR[portNum] = 1<<bitPosi;
    }
    return;
}

uint32_t GPIOGetPinValue( uint32_t portNum, uint32_t bitPosi )
{
  return (GPIO->PIN[portNum] & (1<<bitPosi));
}

/* It is recommended to call this function on system boot or transfer to
   the app. */
void CleanUpSystem(void)
{
    /* Disable peripheral clocks for host IF */
    switch (slIfConfig.ifSel) 
    {
    case SL_I2C0:
        shutdownInterfaceI2C(0);
    
    default:
        break;
    }
    
    CLOCK_DisableClock(kCLOCK_Gpio0);
    CLOCK_DisableClock(kCLOCK_Iocon);
    
    SYSCON->MAINCLKSEL        = 0;     
    SYSCON->MAINCLKUEN        = 0;                  // Toggle update register
    SYSCON->MAINCLKUEN        = 1;    
}


Bool boot_App_ImageCheck(uint8_t app_number)
{
    IMG_HEADER_T *pImageHeader;
    Bool boot = FALSE;
    
    if(app_number==1) {
        pImageHeader = pImageHeader1;
    }
    else {
        pImageHeader = pImageHeader2;
    }
        
    /* check if there is a image header in flash or else don't boot */
    if ((pImageHeader->header_marker == IMG_HEADER_MARKER) &&
        (pinCheckValidHostif(&pImageHeader->hostifPinCfg) != FALSE)) {    
        /* Check if a valid ping config information is programmed in the image header */

        /* imag_type==0---Normal image check IRQ line to halt boot */
        switch (pImageHeader->hostifPinCfg.img_type) 
        {
        case IMG_AP_WAIT:
            /* goto secondary boot loader and wait for AP BOOT command */
            boot = FALSE;
            break;

        default:
        case IMG_NO_CRC:
        case IMG_NORMAL:
            /* check if host wants us to bypass boot due to bad image */
            /* check host requesting us to halt boot through IRQ line (pulled low after reset) */

            GPIOSetDir(PINCFG_GET_PORT(pImageHeader->hostifPinCfg.hostIrqPortPin), PINCFG_GET_PIN(pImageHeader->hostifPinCfg.hostIrqPortPin),0 );
        
            if(GPIOGetPinValue( PINCFG_GET_PORT(pImageHeader->hostifPinCfg.hostIrqPortPin),PINCFG_GET_PIN(pImageHeader->hostifPinCfg.hostIrqPortPin)) == false) {
                /* go to secondary boot loader */
                boot = FALSE;
                break;
            }
            /* skip CRC check if image type is IMG_NO_CRC */
            if (pImageHeader->hostifPinCfg.img_type == IMG_NO_CRC) {
                /* no more checks just boot */
                boot = TRUE;
                break;
            }

        case IMG_NO_WAIT:
            /* Perform a CRC check of the FLASH where the application is located. Jump
               to it if it is valid, but only if this wasn't called from the
               application. */
            if (checkAppCRC(app_number) == 0) {
                /* Valid application exists in application FLASH, so it's ok to boot the
                   application. */                
                boot = TRUE;
            }
            break;
        }
    }
    return boot;

}

void boot_ImageCheck(void)
{
    /*If IRQ pin is pulled low by host, skip app CRC check and don't boot app.*/
    App_CRC_Flag1 = boot_App_ImageCheck(1);
    App_CRC_Flag2 = boot_App_ImageCheck(2);
    
    if(App_CRC_Flag1 == TRUE && App_CRC_Flag2 == TRUE) {
        if(*FW_VERSION1 >= *FW_VERSION2) {
            doCleanBoot(SL_BOOTAPP_ADDR1);
        }
        else {
            doCleanBoot(SL_BOOTAPP_ADDR2);
        }
    }
    else if(App_CRC_Flag1 == TRUE) {
        doCleanBoot(SL_BOOTAPP_ADDR1);
    }
    else if(App_CRC_Flag2 == TRUE) {
        doCleanBoot(SL_BOOTAPP_ADDR2);
    }
}

/*****************************************************************************
 * Public functions
 ****************************************************************************/

 void doCleanBoot(uint32_t appBootAddr)
{
    /* Cleanup before going to app */
    CleanUpSystem();

    bootValidApp(appBootAddr);
    /***** Control should never come here *****/

}

static uint8_t isbootFromReset(void)
{
    /* Was loader booted from app (1) or from FLASH on reset (0x44)? */
    if((uint8_t)*pushAppFlag == 0x44) {
        return 1;
    }

    return 0;
}

static void I2C_Interface_Detect(void)
{
    if (isbootFromReset()) {    
        
        /* Host interface pin setup not provided by app */        
        /*Configurate IRQ pin, P1_14*/            
        IRQ_Pin_Config(&slIfConfig.hostIrqPortPin);

        /*Dectect which I2C is used*/   
        slIfConfig.ifSel = SL_I2C0;
    
    }
    else {
        if(pinCheckValidHostif(pinCfgData)) {
            /* App provided the host interface pin setup, so use it to set the version this app uses */           
            slIfConfig.ifSel = pinCfgData->ifSel;

            /* App defined pins--IRQ pin*/
            parsePortData(&slIfConfig.hostIrqPortPin, pinCfgData->hostIrqPortPin);
        }
    }
    
    /* Set IRQ Pin as OUTPUT */
    GPIOSetDir( slIfConfig.hostIrqPortPin.port, slIfConfig.hostIrqPortPin.pin, 1 );
    Hostif_AssertIRQ();
}

static uint8_t I2C_Interface_Config()
{
    uint8_t i2c_num=0;

    switch (slIfConfig.ifSel) {
		
    case SL_I2C0:
        i2c_num = 0;
        setupMuxingI2C(i2c_num);
        setupInterfaceI2C(i2c_num);
        startHostIfI2C(i2c_num);
        loopI2C_pFunc = &loopHostIfI2C;
        slIfConfig.pI2C = I2C0;
        break;
    }
    return i2c_num;
}


/**
 * @brief    main routine for Secondary loader
 * @return    Function should not exit
 */

int main(void)
{
    IRQn_Type irqNum;
    loopHostIfFunc loopHostIf;    
    int pNum;
    
    BOARD_InitBootPins();
    
    GPIO_PortInit(GPIO, 0);
    GPIO_PortInit(GPIO, 1);
    LED_BLUE_OFF();  
    LED_GREEN_ON();
    
    /*if boot is from reset, check is there valid app? If app is valid, then boot app.*/
    if (isbootFromReset()) {              
        /*If IRQ pin is pulled low by host, skip app CRC check and don't boot app.*/
        boot_ImageCheck();
    }
    
    /* Disable sysTick */
    SysTick->CTRL &= ~SysTick_CTRL_ENABLE_Msk;
    
    /* IAP flash command need to disable the flash cache */
    SYSCON->FLASHCACHECFG = 0;

    /* Disable all NVIC interrupts in case this loader is called from the user application */       
    for (irqNum = SPI0_IRQn; irqNum <= PIN_INT7_IRQn; irqNum++) {
        NVIC_DisableIRQ(irqNum);
    }

    /* Enable core interrupts */
    __enable_irq();

    /* Update core clock variable, main_clk = fro_clk = 30M */
    SystemCoreClockUpdate();

    /* Initialize host interface , configure IRQ pin*/
    Hostif_Init();
    
    /*Detect which I2C is used*/
    I2C_Interface_Detect();

    /*Configurate I2C Interface*/
    pNum = I2C_Interface_Config();

    LED_BLUE_ON();
    
    /********************************************************/
    /****** Start of main host interface handling */
    /********************************************************/

    /* Wait for packets from the host, process the packet, and then respond to the host */
    while (1) {
        
        loopI2C_pFunc(pNum);

        /* Is packet pending? */
        if (hostIfPacketPending() == true) {
            buffXferPending=0;
            /* Send packet to parser and get new packet */
            processHostIfPacket(&recvBuff, &sendBuff);

            /* Handle response, waiting for the rising edge of SPI CS */
            while (hostIfPacketPending() != true) {
                loopI2C_pFunc(pNum);
            }
        }
    }

    /********************************************************/
    /****** End of main host interface handling */
    /********************************************************/

    /* Never called */
    return 0;
}








